#include <qframe.h>
#include <qpainter.h>
#include <qpopupmenu.h>
#include <qtooltip.h>
#include <qtimer.h>
#include <qtextstream.h>
#include <qstringlist.h>
#include <qlistbox.h>
#include <qdatetime.h>

#include <kapp.h>
#include <kmessagebox.h>
#include <kglobal.h>
#include <klocale.h>
#include <kconfig.h>
#include <kprocess.h>
#include <kpanelapplet.h>

extern "C"
{
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/fcntl.h>
#include <sys/types.h>
}

#include "diamon.h"
#include "diamon.moc"


extern "C"
{
	KPanelApplet* init(QWidget *parent, const QString& configFile)
	{
		KGlobal::locale()->insertCatalogue("diamon");
		return new DiaMon(configFile, KPanelApplet::Normal, KPanelApplet::About | KPanelApplet::Help | KPanelApplet::Preferences, parent, "diamon");
	}
}

DiaMon::DiaMon(const QString& configFile, Type type, int actions, QWidget *parent, const char *name)
	:KPanelApplet(configFile, type, actions, parent, name),resizeMode(false),dsize(0),size(60),numMenuItems(3),numProviders(0),onlineStatus(0),timersTimeoutTime(1000),loadScale(12000)
{

#ifdef ERROR_BOX
	/* error box for debug messages */
	ErrBox = new QListBox();
	ErrBox->resize(300,400);
	ErrBox->setCaption("DiaMon's ugly Error Box");
	ErrBox->hide();
#endif


	/* Popup menu */
	menu=new QPopupMenu();
	menu->insertItem(i18n("Help"),this,SLOT(help()),0,0);
	menu->insertItem(i18n("About"),this,SLOT(about()),0,1);
	menu->insertItem(i18n("Settings..."),this,SLOT(preferences()),0,2);
#ifdef ERROR_BOX
	menu->insertItem(i18n("show ErrBox"),ErrBox,SLOT(show()),0,3);
	menu->insertItem(i18n("hide ErrBox"),ErrBox,SLOT(hide()),0,4);
	menu->insertItem(i18n("clear ErrBox"),ErrBox,SLOT(clear()),0,5);
	numMenuItems=6;
#endif
	menu->insertSeparator();
	menu->setCheckable(true);
	QObject::connect( menu, SIGNAL(activated(int)), this, SLOT(toggleConnection(int)) );

	/* preferences dialog */
	pref=new prefDialog(0,i18n("Diamon Settings"),this->config(),MAX_SIZE,MIN_SIZE);
	pref->hide();
	QObject::connect( pref, SIGNAL(okPressed(bool)),this, SLOT(slotPrefOk(bool)) );

	/* get orientation */
	vertical=(orientation()==Qt::Vertical)?true:false;

	/* www connect / disconnect process */
	proc=new KProcess();
	if(! QObject::connect( proc, SIGNAL(processExited(KProcess*) ), this, SLOT(procExited(KProcess*)) ))
		KMessageBox::information(0, "connect() error! (2)");

	/* Timer for online lookup */
	lookupTimer=new QTimer(this);
	QObject::connect( lookupTimer, SIGNAL(timeout()), this, SLOT(lookupLoad()) );
	lookupTimer->start(timersTimeoutTime,false);

	/* device load list */
	loadList.setAutoDelete(true);

	/* get config */
	getDMConfig();

	/* open /dev/isdninfo */
	fp=open("/dev/isdninfo",O_RDONLY | O_NDELAY);

#ifdef ERROR_BOX
	if(fp < 0)	ErrBox->insertItem("cannot open /dev/isdninfo");
#endif

	for(int i=0;i<ISDN_MAX;i++)
	{
		flags[0]=0;
		usage[0]=0;
#ifdef USE_PHONE
		phone[0][0]=0;
#endif
	}

	/* initialize startTime and endTime */
	endTime=startTime=QDateTime::currentDateTime();

	/* set Tooltip font */
	QFont f("courier", 10, QFont::Normal,false);
	f.setFixedPitch(true);
	QToolTip::setFont(f);

#ifdef ERROR_BOX
	ErrBox->insertItem("DiaMon started");
#endif
}

DiaMon::~DiaMon()
{
	loadList.clear();
	delete lookupTimer;
	delete proc;
	delete pref;
	delete menu;
#ifdef ERROR_BOX
	delete ErrBox;
#endif
}

void DiaMon::about()
{
	KMessageBox::information(0, i18n("DiaMon - KDE Web Dialer and Monitor\nVersion "VERSION"\nby Matthias Lieber\n<Mali_@web.de>"));
}

void DiaMon::help()
{
	KMessageBox::information(0, i18n("For DiaMon help look at the README file\n or visit http://www.schneeflocke.net/kde/diamon/"));
}

void DiaMon::preferences()
{
	/* update size for prefDialog, maybe it was changed with MMB */
	pref->changeSize(size);
	/* show prefDialog */
	pref->show();
}

int DiaMon::widthForHeight(int h) const
{
	// horizontal
 	return size;
}

int DiaMon::heightForWidth(int w) const
{
	// vertical
	return size;
}

void DiaMon::resizeEvent( QResizeEvent* r)
{
	// get new size
	height=(r->size()).height();
	width=(r->size()).width();
}

void DiaMon::slotPrefOk(bool updateMenu)
{

	if(!updateMenu)
	{
		if(!updateDMConfig())
			updateMenu=true;
	}

	if(updateMenu)
	{

		// remove all providers from the menu
		for(int i=numMenuItems;i<numMenuItems+numProviders;i++)
		{
			menu->removeItem(i);
		}

		// get new config and insert the new providers to the menu
		getDMConfig();
	}


	updateLayout();
	update();
}


void DiaMon::lookupIfOnline(void)
{
	if(IsdnInfo())
	{
		// read error -> no online status change
		return;
	}

	int old=onlineStatus;
	onlineStatus=0;

	for(int i=0;i<ISDN_MAX;i++)
		if(usage[i])	onlineStatus=1;
	if(onlineStatus)
	{
		for(int i=0;i<ISDN_MAX;i++)
			if(usage[i] && flags[i] && flags[i]!='?')	onlineStatus=2;
	}

	if(onlineStatus==2 && old!=2)
	{
		/* got online */
		startTime=QDateTime::currentDateTime();
	}
	else if(old==2 && onlineStatus!=2)
	{
		/* got offline */
		endTime=QDateTime::currentDateTime();
	}
}

int DiaMon::getLoad(bool init=0)
{
	deviceLoad *tmp;
	char *file="/proc/net/dev";
	char buf[8192];
	unsigned long devStat[16];
	char *c;
	FILE *filep;

	if(loadList.isEmpty())	return 1;

	filep=fopen(file,"rt");
	if(filep==NULL)	return 2;
	if(!fread(buf,sizeof(char),sizeof(buf),filep))	return 3;
	fclose(filep);

	if(tmp=loadList.first()) do
	{
		if((c=strstr(buf,tmp->name))==NULL)
		{
			// device not found
			tmp->ok=false;
			devStat[0]=devStat[8]=0;
		}
		else
		{
			tmp->ok=true;
			c+=strlen(tmp->name)+1;
			sscanf(c,"%ld %ld %ld %ld %ld %ld %ld %ld %ld %ld %ld %ld %ld %ld %ld %ld",
				&devStat[0] , &devStat[1] , &devStat[2] , &devStat[3] ,
				&devStat[4] , &devStat[5] , &devStat[6] , &devStat[7] ,
				&devStat[8] , &devStat[9] , &devStat[10], &devStat[11],
				&devStat[12], &devStat[13], &devStat[14], &devStat[15]);
		}

		if(!init)
		{
			tmp->in[currentLoad] =((devStat[0]-tmp->totalIn )/timersTimeoutTime)*1000;
			tmp->out[currentLoad]=((devStat[8]-tmp->totalOut)/timersTimeoutTime)*1000;
		}
		tmp->totalIn=devStat[0];
		tmp->totalOut=devStat[8];
	}
	while(tmp=loadList.next());

	return 0;
}

void DiaMon::lookupLoad(void)
{
	int e;
	deviceLoad *tmp;

	currentLoad=(currentLoad>=LOAD_ARRAY_SIZE)?0:currentLoad+1;
	if(e=getLoad())
	{
		// failed -> set load (in[],out[]) to 0
		if(tmp=loadList.first()) do
		{
			tmp->in[currentLoad] =0;
			tmp->out[currentLoad]=0;
		}
		while(tmp=loadList.next());
#ifdef ERROR_BOX
		QString ErrMsg;
		if(e==1)		ErrMsg="loadList empty";
		else if(e==2)	ErrMsg="cannot open /proc/net/dev";
		else if(e==3)	ErrMsg="cannot read /proc/net/dev";
		else			ErrMsg="error:  getLoad()";
		ErrBox->insertItem(ErrMsg);
#endif
	}
	lookupIfOnline();
	update();

	QString msg;
	bool nline=false;
	int byteLen=7;
	int sec,min,hour;

	if(onlineStatus==2)
		sec=startTime.secsTo(QDateTime::currentDateTime());
	else
		sec=startTime.secsTo(endTime);

	min=sec/60;
	sec=sec%60;
	hour=min/60;
	min=min%60;

	msg.sprintf("%d:%02d:%02d\n",hour,min,sec);

	if(tmp=loadList.first()) do
	{
		if(nline)	msg+="\n";
		else		nline=true;
		msg+=QString(tmp->name)+" / "+QString(tmp->provider)+":\n";
		if(tmp->ok)
		{
			msg+="  in: "+bytesToStr(tmp->in[currentLoad],byteLen)+"/s ";
			msg+="total:"+bytesToStr(tmp->totalIn,byteLen)+"\n";
			msg+="  out:"+bytesToStr(tmp->out[currentLoad],byteLen)+"/s ";
			msg+="total:"+bytesToStr(tmp->totalOut,byteLen);
		}
		else
		{
			msg+="  not found in /proc/net/dev";
		}
	}
	while(tmp=loadList.next());

	QToolTip::add(this,msg);
}

QString DiaMon::bytesToStr(unsigned long bytes,int len)
{
	QString str;
	int c=0;
	float fbytes=(float)bytes;

	while(fbytes>=1024 && c<6)
	{
		 fbytes/=1024;
		 c++;
	}

	if(!c)
	{
		str=QString::number(bytes);
	}
	else
	{
		str=QString::number(fbytes,'f',1);
		QString a;

		switch(c)
		{
			case 1: a="k"; break;
			case 2: a="M"; break;
			case 3: a="G"; break;
			case 4: a="T"; break;
			case 5: a="P"; break;
			case 6: a="E"; break;
			default: a=""; break;
		}
		str+=a;
	}

	str+="B";
	while(str.length()<len)
		str.prepend(" ");

	return str;
}

int DiaMon::IsdnInfo(void)
{
	char buf[8192];


  		if(read(fp, buf, sizeof(buf)) > 0)
		{
			sscanf(strstr(buf, "usage:"),
         		"usage: %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
         		&usage[0], &usage[1], &usage[2], &usage[3],
         		&usage[4], &usage[5], &usage[6], &usage[7],
         		&usage[8], &usage[9], &usage[10], &usage[11],
	   			&usage[12], &usage[13], &usage[14], &usage[15]);
    		sscanf(strstr(buf, "flags:"),
         		"flags: %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
         		&flags[0], &flags[1], &flags[2], &flags[3],
         		&flags[4], &flags[5], &flags[6], &flags[7],
         		&flags[8], &flags[8], &flags[10], &flags[11],
         		&flags[12], &flags[13], &flags[14], &flags[15]);
#ifdef USE_PHONE
    		sscanf(strstr(buf, "phone:"),
         		"phone: %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s",
         		phone[0], phone[1], phone[2], phone[3],
         		phone[4], phone[5], phone[6], phone[7],
         		phone[8], phone[8], phone[10], phone[11],
         		phone[12], phone[13], phone[14], phone[15]);
#endif
		}
		else
		{
			return 1;
		}

	return 0;
}


void DiaMon::toggleConnection(int MenuID)
{
	if(MenuID<numMenuItems)
		return;
	int ID=MenuID-numMenuItems;
	QString c;

	(this->config())->setGroup("Provider");
	if((this->config())->readNumEntry("Number",0)<=ID)
		return;

	QString s;
	QTextOStream(&s) << "prov" << ID;
	(this->config())->setGroup(s);

	if(menu->isItemChecked(MenuID))
	{
		/* disconnect */
		QString command=(this->config())->readEntry("CommandToDisconnect");
		if(command.isNull())
			return;

		proc->clearArguments();

		QStringList commandList=QStringList::split(" ",command,false);
  		for(int i=0;i<commandList.count();i++)
			(*proc) << commandList[i];

		proc->start(KProcess::NotifyOnExit, KProcess::Stderr);

		menu->setItemChecked (MenuID, false);
	}
	else
	{
		/* connect */
		QString command=(this->config())->readEntry("CommandToConnect");
		if(command.isNull())
			return;

		proc->clearArguments();

		QStringList commandList=QStringList::split(" ",command,false);
  		for(int i=0;i<commandList.count();i++)
			(*proc) << commandList[i];

		proc->start(KProcess::NotifyOnExit, KProcess::Stderr);

		menu->setItemChecked (MenuID, true);
	}

	lookupIfOnline();
}

void DiaMon::procExited(KProcess *p)
{
	int exitStat=p->exitStatus();
	if(exitStat)
	{
		QString msg;
		msg.setNum(exitStat);
		msg.prepend(i18n("Error!\nThe process exited with "));
		KMessageBox::information(0,msg);
	}
}


void DiaMon::paintEvent( QPaintEvent* )
{
 	QPixmap xpm(width,height);
	QPainter p;
	QString w,h;

	if(onlineStatus==2)
		xpm.fill(onC);
	else if(onlineStatus==1)
		xpm.fill(dialC);
	else
		xpm.fill(offC);


    p.begin(&xpm,this);
    p.setBrush(NoBrush);
	p.setPen(QPen(QColor(0,0,0),1));
	bool swap=false;

	if(resizeMode)
		p.drawText(1, 9, QString::number(size));
	else
	{
		if(!(loadList.isEmpty()) && loadScale)
		{
			deviceLoad *tmp;
			int idx=currentLoad;
			int v=(vertical)?width:height;
			int valueIn,valueOut;
			int vtop,vbot,ovtop,ovbot;

			for(int l=size-1;l>=0;l--)
			{
				ovtop=ovbot=0;
				tmp=loadList.first();
				do
				{
					valueIn =( (tmp->in[idx] )*v )/loadScale;
					valueOut=( (tmp->out[idx])*v )/loadScale;

					vtop=( swap)?valueIn:valueOut;
					vbot=(!swap)?valueIn:valueOut;

					if(vertical)
					{
						p.setPen(QPen(( swap)?tmp->inColor:tmp->outColor,1));
						p.drawLine(ovtop,l,vtop+ovtop,l);
						p.setPen(QPen((!swap)?tmp->inColor:tmp->outColor,1));
						p.drawLine(width-(vbot+ovbot),l,width-ovbot,l);
					}
					else
					{
						p.setPen(QPen((!swap)?tmp->inColor:tmp->outColor,1));
						p.drawLine(l,height-(vbot+ovbot),l,height-ovbot);
						p.setPen(QPen(( swap)?tmp->inColor:tmp->outColor,1));
						p.drawLine(l,ovtop,l,vtop+ovtop);
					}

					ovtop+=vtop;
					ovbot+=vbot;

				}
				while(tmp=loadList.next());

				idx=(idx)?idx-1:LOAD_ARRAY_SIZE-1;
			}
		}
	}



    p.flush();
    bitBlt(this,0,0,&xpm);

    p.end();
}

void DiaMon::mousePressEvent(QMouseEvent *event)
{
	/* RMB Menu */
	if(event->button() == RightButton)
		menu->exec( QCursor::pos() );

	/* Resize on MMB down */
	if(event->button() == MidButton)
	{
		resizeMode=true;
		dsize=(vertical)?event->y():event->x();
	}
}

void DiaMon::mouseMoveEvent(QMouseEvent *event)
{
	if(resizeMode)
	{
		/* change size of DiaMon */
		size+=(vertical)?dsize-event->y():dsize-event->x();
		size=(size>MAX_SIZE)?MAX_SIZE:size;
		size=(size<MIN_SIZE)?MIN_SIZE:size;

		updateLayout();
		repaint();

		if(pref->isVisible())
			pref->changeSize(size);
	}

}

void DiaMon::mouseReleaseEvent(QMouseEvent *event)
{
	resizeMode=false;
	(this->config())->setGroup("General");
	(this->config())->writeEntry("Size",size);
	(this->config())->sync();
}


void DiaMon::orientationChange(Orientation o)
{
	vertical=(o==Qt::Vertical)?true:false;
}

void DiaMon::getDMConfig(void)
{
	QColor defin(255,255,255);
	QColor defout(0,0,0);
	loadList.clear();
	currentLoad=0;

	(this->config())->setGroup("Provider");
	int count=(this->config())->readNumEntry("Number",0);
	numProviders=count;
	for(int i=0;i<count;i++)
	{
		QString s;
		QTextOStream(&s) << "prov" << i;
		(this->config())->setGroup(s);

		menu->insertItem( (this->config())->readEntry("Name","Default"),i+numMenuItems);
		loadList.append(new deviceLoad);
		for(int j=0;j<LOAD_ARRAY_SIZE;j++)
		{
			(loadList.current())->in[j]=0;
			(loadList.current())->out[j]=0;
		}
		(loadList.current())->totalIn=0;
		(loadList.current())->totalOut=0;
		(loadList.current())->ok=false;

		strcpy((loadList.current())->name,((this->config())->readEntry("Device")).latin1());
		if(*((loadList.current())->name)=='\0')
		{
			// empty device name
			strcpy((loadList.current())->name,"?");
		}
		strcpy((loadList.current())->provider,((this->config())->readEntry("Name")).latin1());
		(loadList.current())->inColor =(this->config())->readColorEntry("InColor",&defin);
		(loadList.current())->outColor=(this->config())->readColorEntry("OutColor",&defout);
	}

	getLoad(1);

	(this->config())->setGroup("General");
	size=(this->config())->readNumEntry("Size",100);
	loadScale=(this->config())->readNumEntry("Scale",12000);
	timersTimeoutTime=(this->config())->readNumEntry("Timer",1000);
	lookupTimer->changeInterval(timersTimeoutTime);

	QColor def(0,255,0);

	offC=(this->config())->readColorEntry("offlineColor",&def);
	def=QColor(255,255,0);
	dialC=(this->config())->readColorEntry("dailingColor",&def);
	def=QColor(255,0,0);
	onC=(this->config())->readColorEntry("onlineColor",&def);
}

bool DiaMon::updateDMConfig(void)
{
	(this->config())->setGroup("Provider");
	int count=(this->config())->readNumEntry("Number",0);
	if(numProviders!=count)
		return false;

	QColor defin(0,0,0);
	QColor defout(255,255,255);

	int i=0;
	if(loadList.first()) do
	{
		if(i>=count) return false;		// shouldn't happen
		QString s;
		QTextOStream(&s) << "prov" << i;
		(this->config())->setGroup(s);

		strcpy((loadList.current())->name,((this->config())->readEntry("Device")).latin1());
		if(*((loadList.current())->name)=='\0')
		{
			// empty device name
			strcpy((loadList.current())->name,"?");
		}
		strcpy((loadList.current())->provider,((this->config())->readEntry("Name")).latin1());
		(loadList.current())->inColor =(this->config())->readColorEntry("InColor",&defin);
		(loadList.current())->outColor=(this->config())->readColorEntry("OutColor",&defout);
		i++;
	}
	while(loadList.next());

	(this->config())->setGroup("General");
	size=(this->config())->readNumEntry("Size",100);
	loadScale=(this->config())->readNumEntry("Scale",12000);
	timersTimeoutTime=(this->config())->readNumEntry("Timer",1000);
	lookupTimer->changeInterval(timersTimeoutTime);

	QColor def(0,255,0);
	offC=(this->config())->readColorEntry("offlineColor",&def);
	def=QColor(255,255,0);
	dialC=(this->config())->readColorEntry("dailingColor",&def);
	def=QColor(255,0,0);
	onC=(this->config())->readColorEntry("onlineColor",&def);

	return true;
}

