/*
 * KHeiseReg
 *
 * A utility to search for articles within the Heise register.
 *
 * Copyright (C) 2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <qglobal.h>
#include <qnamespace.h>
#include <qsize.h>
#include <qlayout.h>

#include <kapp.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kstdaction.h>
#include <kstdaccel.h>
#include <kmenubar.h>
#include <ktoolbar.h>
#include <ktoolbarbutton.h>
#include <kstatusbar.h>

#include "global.h"
#include "configure.h"
#include "articleview.h"
#include "appenddlg.h"
#include "kheisereg.h"



KHeiseReg::KHeiseReg(QWidget *parent, const char *name): KMainWindow(parent, name)
{
	config = kapp->config();
	
	initActions();
	initMenuBar();
	initToolBar();
	initStatusBar();
	
	view = new KHeiseView(this, "kheiseview");
	CHECK_PTR(view);
	view->setMinimumSize(view->sizeHint());
	setCentralWidget(view);

	readOptions();

	searchdlg = new SearchDlg(0, "searchdlg");
	CHECK_PTR(searchdlg);
	
	connect(view, SIGNAL(items(int)), SLOT(slotStatusItems(int)));
//	connect(view, SIGNAL(command(int)), SLOT(commandCallback(int)));
	connect(searchdlg, SIGNAL(search(const RegMask &)), SLOT(slotSearch(const RegMask &)));
}


KHeiseReg::~KHeiseReg()
{
}


void KHeiseReg::readOptions()
{
	config->setGroup("MainWindow Options");

	QSize geoSize(704, 506);
	resize(config->readSizeEntry("Geometry", &geoSize));
	settingsShowToolbar->setChecked(config->readBoolEntry("Show Toolbar", true));
	slotSettingsShowToolbar();
	settingsShowStatusbar->setChecked(config->readBoolEntry("Show Statusbar", true));
	slotSettingsShowStatusbar();
	toolBar()->setBarPos((KToolBar::BarPosition)config->readNumEntry("ToolbarPos", (int)KToolBar::Top));
	
	view->readOptions();
}


void KHeiseReg::initActions()
{
  fileOpen = new KAction(i18n("&Append Register File..."), "fileopen", CTRL+Key_A, this, SLOT(slotFileOpen()), actionCollection(), "file_open");
	fileOpen->setWhatsThis(i18n("Click this button to open the dialog for appending register files to the data base.<br><br>" \
		"You can also select the command <b>Append Register File</b> from the \"File\" menu."));
  fileQuit = KStdAction::quit(this, SLOT(slotFileQuit()), actionCollection());
	fileQuit->setWhatsThis(i18n("Click this button to quit KHeiseReg.<br><br>" \
		"You can also select the command <b>Quit</b> from the \"File\" menu."));

  searchSearch = KStdAction::find(this, SLOT(slotSearchSearch()), actionCollection());
	searchSearch->setWhatsThis(i18n("Click this button to open the search dialog.<br><br>" \
		"You can also select the command <b>Search</b> from the \"Search\" menu."));

  settingsShowToolbar = KStdAction::showToolbar(this, SLOT(slotSettingsShowToolbar()), actionCollection());
  settingsShowStatusbar = KStdAction::showStatusbar(this, SLOT(slotSettingsShowStatusbar()), actionCollection());
  settingsSaveOptions = new KAction(i18n("Save &Options"), 0, this, SLOT(slotSettingsSaveOptions()), actionCollection(), "settings_save_options");
  settingsConfigure = new KAction(i18n("&Configure KHeiseReg..."), "configure", 0, this, SLOT(slotSettingsConfigure()), actionCollection(), "settings_configure");
	settingsConfigure->setWhatsThis(i18n("Opens the Configure Dialog.<br><br>" \
		"You can also select the command <b>Configure KHeiseReg</b> from the \"Settings\" menu."));
		
/*
	i18n("Invokes the KDE help system with the KHeiseReg help pages displayed.<br><br>" \
		"You can also select the command <b>Contents</b> from the \"Help\" menu."));
*/
}


void KHeiseReg::initMenuBar()
{
	QPopupMenu *fileMenu, *searchMenu, *settingsMenu;

	fileMenu = new QPopupMenu(0, "filemenu");
	CHECK_PTR(fileMenu);
  fileOpen->plug(fileMenu);
	fileMenu->insertSeparator();
  fileQuit->plug(fileMenu);

	searchMenu = new QPopupMenu(0, "searchmenu");
	CHECK_PTR(searchMenu);
	searchSearch->plug(searchMenu);
	
	settingsMenu = new QPopupMenu(0, "settingsmenu");
	CHECK_PTR(settingsMenu);
  settingsShowToolbar->plug(settingsMenu);
  settingsShowStatusbar->plug(settingsMenu);
	settingsMenu->insertSeparator();
  settingsSaveOptions->plug(settingsMenu);
	settingsMenu->insertSeparator();
  settingsConfigure->plug(settingsMenu);

	menuBar()->insertItem(i18n("&File"), fileMenu);
	menuBar()->insertItem(i18n("S&earch"), searchMenu);
	menuBar()->insertItem(i18n("&Settings"), settingsMenu);
	menuBar()->insertSeparator();
	menuBar()->insertItem(i18n("&Help"), helpMenu());
}


void KHeiseReg::initToolBar()
{
	fileOpen->plug(toolBar());
	toolBar()->insertSeparator();
	searchSearch->plug(toolBar());
	toolBar()->insertSeparator();
	settingsConfigure->plug(toolBar());
	toolBar()->insertSeparator();
	toolBar()->insertButton(BarIcon("help"), ID_HELP_CONTENTS, SIGNAL(clicked()), this, SLOT(appHelpActivated()), true, i18n("Help"));
	toolBar()->alignItemRight(fileQuit->itemId(fileQuit->plug(toolBar())));
}

	
void KHeiseReg::initStatusBar()
{
	statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG, 1);
	statusBar()->setItemAlignment(ID_STATUS_MSG, AlignLeft | AlignVCenter);
	statusBar()->insertFixedItem(i18n(" %1 Articles ").arg(88888), ID_STATUS_ITEMS);
	statusBar()->changeItem("", ID_STATUS_ITEMS);
}


void KHeiseReg::slotFileOpen()
{
	AppendDlg appenddlg(this);
	appenddlg.exec();
}


void KHeiseReg::slotFileQuit()
{
	close();
}


void KHeiseReg::slotSearchSearch()
{
	if (searchdlg->isVisible())
		searchdlg->raise();
	else
		searchdlg->show();
}


void KHeiseReg::slotSettingsShowToolbar()
{
	if (settingsShowToolbar->isChecked())
		toolBar()->show();
	else
		toolBar()->hide();
}


void KHeiseReg::slotSettingsShowStatusbar()
{
	if (settingsShowStatusbar->isChecked())
		statusBar()->show();
	else
		statusBar()->hide();
}


void KHeiseReg::slotSettingsSaveOptions()
{
	config->setGroup("MainWindow Options");
	
	config->writeEntry("Geometry", size());
	config->writeEntry("Show Toolbar", toolBar()->isVisible());
	config->writeEntry("Show Statusbar", statusBar()->isVisible());
	config->writeEntry("ToolbarPos", (int)toolBar()->barPos());

	view->saveOptions();
}


void KHeiseReg::slotSettingsConfigure()
{
	Configure configure(this, "configure");
	configure.exec();
}


void KHeiseReg::slotStatusMsg(const QString &text)
{
	statusBar()->clear();
	statusBar()->changeItem(text, ID_STATUS_MSG);
}


void KHeiseReg::slotStatusHelpMsg(const QString &text)
{
	statusBar()->message(text, 2000);
}


void KHeiseReg::slotStatusItems(int items)
{
	statusBar()->changeItem(i18n(" %1 Articles ").arg(KGlobal::locale()->formatNumber((double)items, 0)), ID_STATUS_ITEMS);
	slotStatusMsg(i18n("Ready."));
}
	

void KHeiseReg::slotSearch(const RegMask &mask)
{
	slotStatusMsg(i18n("Searching..."));
	view->search(mask);
}
